require('dotenv').config();
const { Client, GatewayIntentBits, Events, PermissionFlagsBits } = require('discord.js');

const client = new Client({
  intents: [
    GatewayIntentBits.Guilds,
    GatewayIntentBits.GuildMembers,
    GatewayIntentBits.GuildModeration,
    GatewayIntentBits.GuildVoiceStates,
  ],
});

const GUILD_ID = process.env.GUILD_ID;
const LOG_LEVEL = process.env.LOG_LEVEL || 'info';

function log(level, ...args) {
  const levels = { debug: 0, info: 1, warn: 2, error: 3 };
  const configLevel = levels[LOG_LEVEL] || 1;
  if (levels[level] >= configLevel) {
    console.log(`[${level.toUpperCase()}]`, ...args);
  }
}

client.once(Events.ClientReady, (c) => {
  log('info', `✅ Bot ready! Logged in as ${c.user.tag}`);
  log('info', `Guild ID: ${GUILD_ID}`);
  log('info', `Serving ${c.guilds.cache.size} guild(s)`);
  
  // Verify guild access
  const guild = c.guilds.cache.get(GUILD_ID);
  if (guild) {
    log('info', `✅ Connected to guild: ${guild.name}`);
    log('info', `Bot has ${guild.roles.cache.size} roles visible`);
  } else {
    log('warn', `⚠️  Guild ${GUILD_ID} not found. Ensure bot is invited to the server.`);
  }
});

// Role management events (for audit/logging)
client.on(Events.GuildMemberUpdate, (oldMember, newMember) => {
  const addedRoles = newMember.roles.cache.filter(role => !oldMember.roles.cache.has(role.id));
  const removedRoles = oldMember.roles.cache.filter(role => !newMember.roles.cache.has(role.id));
  
  if (addedRoles.size > 0) {
    log('debug', `Role(s) added to ${newMember.user.tag}:`, addedRoles.map(r => r.name).join(', '));
  }
  if (removedRoles.size > 0) {
    log('debug', `Role(s) removed from ${newMember.user.tag}:`, removedRoles.map(r => r.name).join(', '));
  }
});

// Member join/leave logging
client.on(Events.GuildMemberAdd, (member) => {
  log('info', `Member joined: ${member.user.tag} (${member.id})`);
});

client.on(Events.GuildMemberRemove, (member) => {
  log('info', `Member left: ${member.user.tag} (${member.id})`);
});

// Error handling
client.on(Events.Error, (error) => {
  log('error', 'Discord client error:', error);
});

process.on('unhandledRejection', (error) => {
  log('error', 'Unhandled promise rejection:', error);
});

// Graceful shutdown
process.on('SIGINT', () => {
  log('info', 'Shutting down bot...');
  client.destroy();
  process.exit(0);
});

process.on('SIGTERM', () => {
  log('info', 'Shutting down bot...');
  client.destroy();
  process.exit(0);
});

// Login
if (!process.env.DISCORD_TOKEN) {
  log('error', '❌ DISCORD_TOKEN not found in .env file');
  process.exit(1);
}

if (!GUILD_ID) {
  log('error', '❌ GUILD_ID not found in .env file');
  process.exit(1);
}

client.login(process.env.DISCORD_TOKEN).catch((error) => {
  log('error', '❌ Failed to login:', error.message);
  process.exit(1);
});
